const float PI = radians(180.0);
const float TAU = radians(360.0);
const float HPI = PI * 0.5;
const float rPI = 1.0 / PI;
const float rTAU = 1.0 / TAU;
const float PHI = sqrt(5.0) * 0.5 + 0.5;

float saturate(float x)
{
	return clamp(x, 0.0, 1.0);
}

vec3 saturate(vec3 x)
{
	return clamp(x, vec3(0.0), vec3(1.0));
}

vec2 saturate(vec2 x)
{
	return clamp(x, vec2(0.0), vec2(1.0));
}

vec2 EncodeNormal(vec3 normal)
{
	float p = sqrt(normal.z * 8.0 + 8.0);
	return vec2(normal.xy / p + 0.5);
}

vec3 DecodeNormal(vec2 enc)
{
	vec2 fenc = enc * 4.0 - 2.0;
	float f = dot(fenc, fenc);
	float g = sqrt(1.0 - f / 4.0);
	vec3 normal;
	normal.xy = fenc * g;
	normal.z = 1.0 - f / 2.0;
	return normal;
}


float remap(float e0, float e1, float x)
{
	return saturate((x - e0) / (e1 - e0));
}


vec4 SampleLinear(sampler2D tex, vec2 coord)
{
	return pow(texture2D(tex, coord), vec4(2.2));
}

vec3 LinearToGamma(vec3 c)
{
	return pow(c, vec3(1.0 / 2.2));
}

vec3 GammaToLinear(vec3 c)
{
	return pow(c, vec3(2.2));
}

float curve(float x)
{
	return x * x * (3.0 - 2.0 * x);
}

vec2 curve(vec2 x)
{
	return x * x * (3.0 - 2.0 * x);
}

vec3 curve(vec3 x)
{
	return x * x * (3.0 - 2.0 * x);
}

float Luminance(in vec3 color)
{
	return dot(color.rgb, vec3(0.2125f, 0.7154f, 0.0721f));
}

vec3 rand(vec2 coord)
{
	float noiseX = saturate(fract(sin(dot(coord, vec2(12.9898, 78.223))) * 43758.5453));
	float noiseY = saturate(fract(sin(dot(coord, vec2(12.9898, 78.223)*2.0)) * 43758.5453));
	float noiseZ = saturate(fract(sin(dot(coord, vec2(12.9898, 78.223)*3.0)) * 43758.5453));

	return vec3(noiseX, noiseY, noiseZ);
}

const float rotationArray[32] = float[32](
    0.0,
    3.883222,
    7.766444,
    11.649666,
    15.532888,
    19.41611,
    23.299332,
    27.182554,
    31.065776,
    34.948998,
    38.83222,
    42.715442,
    46.598664,
    50.481886,
    54.365108,
    58.24833,
    62.131552,
    66.014774,
    69.897996,
    73.781218,
    77.66444,
    81.547662,
    85.430884,
    89.314106,
    93.197328,
    97.08055,
    100.963772,
    104.846994,
    108.730216,
    112.613438,
    116.49666,
    120.379882
);

float LDdither(vec2 coord)
{
	return saturate(fract(dot(coord, vec2(1.0, 1.0))) * 500.0);
}

/*
	    colorEdge += (DoFXAASimple(gaux3, coord.st + vec2( texel.x,  texel.y), 1.0 / vec2(viewWidth, viewHeight)).rgb);
	    colorEdge += (DoFXAASimple(gaux3, coord.st + vec2( texel.x, -texel.y), 1.0 / vec2(viewWidth, viewHeight)).rgb);
	    colorEdge += (DoFXAASimple(gaux3, coord.st + vec2(-texel.x,  texel.y), 1.0 / vec2(viewWidth, viewHeight)).rgb);
	    colorEdge += (DoFXAASimple(gaux3, coord.st + vec2(-texel.x, -texel.y), 1.0 / vec2(viewWidth, viewHeight)).rgb);	
	    colorEdge -= color;
	    colorEdge /= 3.0;

	float hcc = mix(1.0, 0.1, exp(-Luminance(color)));
		hcc = 0.25;
		hcc = 1.0
		 * pow(Luminance(color), hcc)
		;

	color += clamp(dot(color - colorEdge, vec3(.3333333)), -4e-3, 4e-3) * SHARPENING * hcc * normalize(color.rgb + 1e-9);  
*/	

vec3 noised(vec2 coord)
{
	float noiseX = saturate(fract((dot(coord, vec2(0.9898, 1.223))) * 43758.5453));
	float noiseY = saturate(fract((dot(coord, vec2(0.9898, 1.223)*2.0)) * 43758.5453));
	float noiseZ = saturate(fract((dot(coord, vec2(0.9898, 1.223)*3.0)) * 43758.5453));

	return vec3(noiseX, noiseY, noiseZ);
}

float SmoothMin( float a, float b, float k )
{
    float h = clamp( 0.5+0.5*(b-a)/k, 0.0, 1.0 );
    return mix( b, a, h ) - k*h*(1.0-h);
}

float SmoothMax( float a, float b, float k )
{
    return -SmoothMin(-a, -b, k);
}

vec4 ToSH(float value, vec3 dir)
{
	const float N1 = sqrt(4 * PI / 3);
	const float transferl1 = (sqrt(PI) / 3.0) * N1;
	const float transferl0 = PI;

	const float sqrt1OverPI = sqrt(1.0 / PI);
	const float sqrt3OverPI = sqrt(3.0 / PI);

	vec4 coeffs;

	coeffs.x = 0.5 * sqrt1OverPI * value * transferl0;
	coeffs.y = -0.5 * sqrt3OverPI * dir.y * value * transferl1;
	coeffs.z = 0.5 * sqrt3OverPI * dir.z * value * transferl1;
	coeffs.w = -0.5 * sqrt3OverPI * dir.x * value * transferl1; //TODO: Vectorize the math so it's faster

	return coeffs;
}


vec3 FromSH(vec4 cR, vec4 cG, vec4 cB, vec3 lightDir)
{
	const float N1 = sqrt(4 * PI / 3);
	const float transferl1 = (sqrt(PI) / 3.0) * N1;
	const float transferl0 = PI;

	const float sqrt1OverPI = sqrt(1.0 / PI);
	const float sqrt3OverPI = sqrt(3.0 / PI);

	vec4 sh;

	sh.x = 0.5 * sqrt1OverPI;
	sh.y = -0.5 * sqrt3OverPI * lightDir.y;
	sh.z = 0.5 * sqrt3OverPI * lightDir.z;
	sh.w = -0.5 * sqrt3OverPI * lightDir.x;

	vec3 result;
	result.r = sh.x * cR.x;
	result.r += sh.y * cR.y;
	result.r += sh.z * cR.z;
	result.r += sh.w * cR.w;

	result.g = sh.x * cG.x;
	result.g += sh.y * cG.y;
	result.g += sh.z * cG.z;
	result.g += sh.w * cG.w;

	result.b = sh.x * cB.x;
	result.b += sh.y * cB.y;
	result.b += sh.z * cB.z;
	result.b += sh.w * cB.w;

	return result.rgb;
}

const float goldenAngle = TAU / PHI / PHI;

const float LOG2 = log(2.0);
const float rLOG2 = 1.0 / LOG2;

const float LOG10  = log(10.0);
const float rLOG10 = 1.0 / LOG10;

float log10(float x)
{
  return log(x)*rLOG10;
}

//x is distance to outer surface, y is distance to inner surface
vec2 RaySphereIntersection( vec3 p, vec3 dir, float r ) 
{
	float b = dot( p, dir );
	float c = dot( p, p ) - r * r;
	
	float d = b * b - c;
	if ( d < 0.0 ) 
	{
		return vec2( 10000.0, -10000.0 );
	}

	d = sqrt( d );
	
	return vec2( -b - d, -b + d );
}


#define R_INNER 0.985

// Mie
// g : ( -0.75, -0.999 )
//      3 * ( 1 - g^2 )               1 + c^2
// F = ----------------- * -------------------------------
//      2 * ( 2 + g^2 )     ( 1 + g^2 - 2 * g * c )^(3/2)
float phase_mie( float g, float c, float cc ) {
	float gg = g * g;
	float mp = (1.0 + cc) * (1.0 - gg) / (2.0 + gg) * pow(1.0 + gg - g * 2.0 * c, -1.5) * 1.5
	//+ g * c * 0.05
	;
	//mp *= 0.125 * rPI;
	return mp;
	//return pow(-2.0 * g * c + gg + 1.0, -1.5) * (1.0 + cc);
}

vec3 phase_mie_col( vec3 g, float c, float cc ) {
	vec3 gg = g * g;
	
	vec3 a = ( 1.0 - gg ) * ( 1.0 + cc );

	vec3 b = 1.0 + gg - 2.0 * g * c;
	b *= sqrt( b );
	b *= 2.0 + gg;	
	
	return 1.5 * a / b;
}

float Greenphase( float g, float c, float cc ) {
	float gg = g * g;
	
	float a = ( 1.0 - gg ) * ( 1.0 + cc );

	float b = 1.0 + gg - 2.0 * g * c;
	b *= sqrt( b );	
	
	return a / b;
}

// Reyleigh
// g : 0
// F = 3/4 * ( 1 + c^2 )
float phase_reyleigh( float cc ) 
{
	return 0.75 * ( 1.0 + cc );
}

float density( vec3 p )
{
	const float R = 1.0;
	const float SCALE_H = 4.0 / ( R - R_INNER );
	const float SCALE_L = 1.0 / ( R - R_INNER );

	return exp( -( length( p ) - R_INNER ) * SCALE_H ) * 2.0;
}

float TotalAtmosphereDensity(float dirY, const bool considerLand)
{
	// return 1.0 / (max(0.0, dirY + 0.06) * 3.0 + 0.00001);

	if (!considerLand || (considerLand && dirY > 0.0))
	{
		return 1.0 / (max(0.0, dirY) * 6.0 + 0.03);
	}
	else
	{
		return 1.0 / (max(0.0, -dirY) * 50.0 + 0.4);
	}
}

float optic( vec3 p, vec3 q ) 
{
	const int numOutscatter = 0;

	const float R = 1.0;
	const float SCALE_L = 1.0 / (R - R_INNER);

	vec3 step = ( q - p ) / float(numOutscatter);
	step *= 0.3;
	vec3 v = p + step * 0.5;
	
	float sum = 0.0;
	for ( int i = 0; i < numOutscatter; i++ ) 
	{
		sum += density( v );
		v += step;
	}
	sum *= length( step ) * SCALE_L;


	return sum;
}

const float worldscale = 1e-5f;
const float FEnergy = 80.0f;
const float Energy = 0.0f;
const vec3 C_R1 = vec3(0.17, 0.4, 0.99);
const vec3 C_R2 = vec3(0.3, 0.7, 0.99);
float Adensity = 0.005;
float Cdensity = 0.07;
const float ColorCurved = 1.0;

vec3 AtmosphereAbsorption(vec3 dir, float depth)
{
	float x = depth;
	float v = dir.y;
	vec3 R = C_R1;
	float M = Adensity;
	float rD = Cdensity;
	float rDF = 3e-1;
	
	float rDFv = rDF * v;

	vec3 absorption = exp((-M - R)*(rD/(rDFv) - rD/(exp(rDFv*x)*(rDFv))));

	if (depth > 10.0 && dir.y < 0.0) {
		absorption *= 0.0;
	}

	return absorption;
}

vec3 SunAbsorptionAtAltitude(vec3 Dir, float altitude)
{
	float a = altitude;
	float s = Dir.y;
	vec3 R = C_R1;
	float M = Adensity;
	float rD = Cdensity;
	float rDF = 3e-1;

	float rDFs = rDF * saturate(s);
	float rDFa = rDF * a;

	return exp(-(R + M) * rD/(exp(rDFa)*(rDFs)) - (rD*exp(-rDFa - 1000.0*rDFs))/(rDFs));
}

vec3 in_scatter(vec3 o, vec3 dir, vec2 e, vec3 l, const float mieAmount, const float rayleighAmount) 
{
	const float numInscatter = 1;

	const float R = 1.0;
	const float SCALE_L = 1.0 / (R - R_INNER);

	const float K_R = 0.186 * rayleighAmount;
	const float K_M = 0.035 * mieAmount;
	const float E = Energy;
	const vec3 C_R = C_R1;	//Rayleigh scattering coefficients

	float boosty = saturate(l.y + 0.1) * 0.95 + 0.05;
	boosty = 1.0 / sin(boosty);

	float len = (e.y * (1.0 + boosty * 0.0)) / float(numInscatter);
	vec3 step = dir * len;
	step *= 2.0;
	vec3 p = o;
	
	float c  = dot( dir, -l + dir );
	float cc = c * c;
	
	dir.y = dir.y - 0.00001;	
	dir = normalize(dir);
	
	float LdotVis = dot(dir, l);
	float LdotV2is = LdotVis * LdotVis;	
	
	float LdotVo = dot(dir, -l) * 0.5 + 0.5;	
	float LdotV = dot(dir, l) * 0.5 + 0.5;
	float LdotV2 = LdotV * LdotV;
	//float boosty = 1.0 - abs(l.y);
	float s = l.y;	
	s = SmoothMax(0.01, s, 0.07);
	vec3 v = p + dir * ( len * (0.5 + boosty * 0.0) );
	float HeigthH = saturate(1.0 - exp2(-300.0 * worldscale * max(mieAmount, 0.0f)));	
	float HeigthD = exp2(-max(dir.y, 0.0f) * worldscale / 1e-4 * max(mieAmount, 0.0f));
		 Cdensity *= HeigthD;
		 Adensity *= HeigthD;
		 
	float v2 = (pow(abs(dir.y), 1.0 + saturate(-LdotV * 0.5 + 0.5) * 0.0005 / (s + 0.0005)) * sign(dir.y));
	float energyFade = exp(min(0.0, l.y) * 50.0);
	float rDF = (0.3 - (-LdotV * 0.5 + 0.5) * max(0.0, -l.y + 0.01) * 4.0);
	//float rDF = 0.3;
	float M = Adensity + Cdensity * exp(-(max(dir.y, 0.0f) * 1.0 + 0.05));
	float t1 = (s + (-1.0 + exp(-1000.0*rDF*s))*v2);	
	vec3 MpR = (Adensity + Cdensity * exp(-(max(dir.y, 0.0f) + 0.05)) + C_R);
	float t3 = (rDF*v2);
	float t3rcp = 1.0 / t3;
	vec3 t4 = (MpR*t1);
	vec3 t4rcp = vec3(1.0) / t4;
	vec3 t5 = (MpR*Cdensity)
	// * exp2(-max(dir.y, 0.0f) * worldscale / 1e-4 * max(mieAmount, 0.0f))	
	// / (max(dir.y, 0.0f) * worldscale / 1e-4 * pow(max(mieAmount, 0.0f), 3.0) * 0.0001 + 1.0f)
	;
	
    //if(length(t5) <= 1e-4) t5 = vec3(0.0);
	
	float x = e.y;
	float floorDist = min(x, 0.5 / (-dir.y + 0.009));
	float floorExsist = 0.0f;
	if (dir.y < 0.0)
	{
		x = floorDist;
		floorExsist = 1.0f;
	}
	
	vec3 blueT = pow(vec3(0.85, 0.85, 1.0), vec3(1.0 / saturate(l.y * 0.7 + 0.3))); 
    vec3 at = (exp((t5*(-1.0 + t1/s))*t3rcp)*s)*t4rcp - (exp((t5*(-1.0 + t1/(exp(t3*x)*s)))*t3rcp)*s)*t4rcp;
    //vec3 at = AtmosphereAbsorption(dir, 80.0);	
	at *= energyFade;
	at *= blueT;
	vec3 ms = 1.0/MpR - 1.0/(exp(((-1.0 + exp(t3*x))*MpR*0.07)/(exp(t3*x)*(t3)))*MpR);
	vec3 erc = vec3(0.);
	//float Ldagc = pow(1.0 + (1.0 - LdotV) * 1e4 * M * 1.3, -1.0);
	float Ldagc = 1.0;		
	float Lagc2 = pow(LdotV * 0.5 + 0.5, 1.2) * Ldagc;	
	float Lagco2 = pow(LdotVo * 0.5 + 0.5, 1.2) * Ldagc;		
	//if (dir.y < 0.0) {
	//	erc += mix(phase_mie(0.0, LdotV, LdotV2) * C_R / (0.01 + max(-dir.y, 0.0f) * 2.0), vec3(0.0), 1.0 / (max(-dir.y, 0.0f) * worldscale / 1e-5 * pow(max(mieAmount, 0.0f), 2.0) * 0.00001 + 1.0f)) * saturate(l.y * 100.0);
	//	erc += mix(C_R * exp(-MpR * floorDist * 0.1) * 0.3 * exp(-MpR * (0.2 / (saturate(l.y) + 0.001))), vec3(0.0), vec3(1.0) / (max(dir.y, 0.0f) * worldscale / 1e-4 * pow(max(mieAmount, 0.0f), 2.0) * 0.2 + 1.0f));
	//}
	return max(vec3(0.0f),at * (0.0
	  + max(0.0, 0.0
	           //+ Greenphase(-Lagco2 * 0.2 * HeigthD, LdotVis, LdotV2is)	
	           + Greenphase(Lagc2 * 0.9 * HeigthD, LdotVis, LdotV2is) )
			   * M * 0.8
	           //+ Greenphase(0.1 * HeigthD, LdotVis, LdotV2is)	 * C_R
	           + (LdotV2 * 0.75 + 0.75) * C_R			   
	  //+ phase_mie(0.9 * LdotV, LdotV, LdotV2) * M * 3.0
	  //+ phase_mie(0.1, LdotV, LdotV2) * C_R * 0.8
	  //+ phase_mie(0.1, LdotV, LdotV2) * C_R
	  //+ phase_mie(0.1 * Lagc2, LdotVis, LdotV2is) * C_R	  
      //+ phase_mie_col(, LdotV, LdotV2 )	  
	  )
	  //+max(vec3(0.0), ms * C_R + ms * 0.06) * exp(SmoothMin(0.0, l.y, 0.03) * 200.0) * 0.03 / (max(dir.y, 0.0f) * worldscale / 1e-4 * pow(max(mieAmount, 0.0f), 3.0) * 0.0001 + 1.0f)
      //+erc
	  + vec3(0.008) * energyFade * blueT * HeigthD
	  ) * E * (saturate(1.0 - mix(c * HeigthH, 0.0, saturate(l.y))) * 0.996 + 0.004);
}

vec3 in_scatter2(vec3 o, vec3 dir, vec2 e, vec3 l) 
{
	const float numInscatter = 1;

	const float R = 1.0;
	const float SCALE_L = 1.0 / (R - R_INNER);

	const float K_R = 0.166;
	const float K_M = 0.00;
	const float E = Energy;
	const vec3 C_R = C_R2;	//Rayleigh scattering coefficients

	float len = (dir.y) / float(numInscatter);
	vec3 step = dir * len;
	step *= 2.0;
	vec3 p = o;
	float c  = dot( dir, -l );
	float cc = c * c;
	float LdotV = dot(dir, l);
	float LdotV2 = LdotV * LdotV;
	//float boosty = 1.0 - abs(l.y);
	float boosty = saturate(l.y + 0.1) * 0.95 + 0.05;
	boosty = 1.0 / sin(boosty);
	float s = l.y;	
	s = SmoothMax(0.01, s, 0.07);
	vec3 v = p + dir * ( len * (0.5 + boosty * 0.0) );
	dir = dir;
	dir = normalize(dir);	
	dir.y = dir.y - 0.001;
	//dir = normalize(dir);
	float v2 = (pow(abs(dir.y), 1.0 + saturate(-LdotV * 0.5 + 0.5) * 0.0005 / (s + 0.0005)) * sign(dir.y));
	float energyFade = exp(min(0.0, l.y) * 100.0);
	
	float rDF = 0.3 - (-LdotV * 0.5 + 0.5) * max(0.0, -l.y + 0.01) * 4.0;
	float t1 = (s + (-1.0 + exp(-1000.0*rDF*s))*v2);
	vec3 MpR = (0.005 + Cdensity / (saturate(dir.y) + 0.05) + C_R);
	float t3 = (rDF*v2);
	float t3rcp = 1.0 / t3;
	vec3 t4 = (MpR*t1);
	vec3 t4rcp = vec3(1.0) / t4;
	vec3 t5 = (MpR*0.07) * Adensity;
	

	vec3 sum = vec3( 0.0 );
	for ( int i = 0; i < numInscatter; i++ ) 
	{
		vec2 f = RaySphereIntersection( v, l, R );
		vec3 u = v + l * f.y;
		
		float n = ( optic( p, v ) + optic( v, u ) ) * ( PI * 4.0 );
		
		sum += density( v ) * exp( -n * ( K_R * C_R + K_M ) );

		v += step;
	}
	sum *= len * SCALE_L;
	

	
	float x = FEnergy;
	float floorDist = min(x, 0.5 / (-dir.y + 0.00918));

	if (dir.y < 0.0)
	{
		x = floorDist;
	}
    vec3 at = (exp((t5*(-1.0 + t1/s))*t3rcp)*s)*t4rcp - (exp((t5*(-1.0 + t1/(exp(t3*x)*s)))*t3rcp)*s)*t4rcp;
	vec3 ms = 1.0/MpR - 1.0/(exp(((-1.0 + exp(t3*x))*MpR*0.07)/(exp(t3*x)*(t3)))*MpR);
	vec3 erc = vec3(0.);
	if (dir.y < 0.0) {
		erc += vec3(0.2, 0.65, 1.0) * exp(-MpR * floorDist * 0.1) * 0.3 * exp(-MpR * (0.2 / (saturate(l.y) + 0.001)));
	}	
	return max(vec3(0.0f), at * (0.0f
	  + phase_mie( .0, LdotV, LdotV2 ) * C_R
	  + phase_mie( .6, LdotV, LdotV2 )	  
	  )
	  * energyFade
	  * pow(C_R, vec3(0.005 / (saturate(l.y) + 0.009)))	
	  +(max(vec3(0.0), ms * C_R + ms * 0.05)  *  exp(SmoothMin(0.0, l.y, 0.03) * 200.0) * 0.01+erc) * 0.0f
	  )* E;
}

vec3 AtmosphericScattering(vec3 rayDir, vec3 lightVector, const float mieAmount)
{
	const float DEG_TO_RAD = PI / 180.0;

	//Scatter constants
	const float K_R = 0.166;
	const float K_M = 0.0025;
	const float E = Energy;
	const vec3 C_R = C_R1;	//Rayleigh scattering coefficients
	const float G_M = -0.85;

	const float R = 1.0;
	const float SCALE_H = 4.0 / (R - R_INNER);
	const float SCALE_L = 1.0 / (R - R_INNER);

	const int NUM_OUT_SCATTER = 10;
	const float FNUM_OUT_SCATTER = 10.0;

	const int NUM_IN_SCATTER = 10;
	const float FNUM_IN_SCATTER = 10.0;

	vec3 eye = vec3(0.0, mix(R_INNER, 1.0, 0.05), 0.0);

	vec3 originalRayDir = rayDir;

	//if (rayDir.y < 0.0)
	{
		//rayDir.y = abs(rayDir.y);
		//rayDir.y *= rayDir.y;
		//rayDir.y = 0.0;
	}

	vec3 up = vec3(0.0, 1.0, 0.0);
    //rayDir.y = rayDir.y * 0.5 + 0.5;
	//rayDir = normalize(rayDir);
	
	vec2 e = RaySphereIntersection(eye, rayDir, R);
	vec2 eup = RaySphereIntersection(eye, up, R);

	e.y = 80.0;
	
	vec3 atmosphere = in_scatter(eye, rayDir, e, lightVector, mieAmount, 1.0);

	vec3 secondary = in_scatter2(eye, up, eup, lightVector);

	vec3 ambient = C_R2;

	vec3 ground = vec3(0.1, 0.1, 0.1) * 0.05;

	float boosty = saturate(lightVector.y) * 0.90 + 0.10;
	boosty = 1.0 / sin(boosty);

	//atmosphere += dot(secondary, vec3(0.06)) * ambient * boosty;
	//atmosphere += dot(secondary, vec3(0.86)) * ambient;
	//atmosphere += ambient * 0.01;

	//atmosphere *= vec3(0.9, 0.9, 1.0);
	//atmosphere *= pow(C_R2, vec3(0.5));
	
	atmosphere = pow(atmosphere, vec3(ColorCurved));
	

	
	//if (originalRayDir.y < 0.0)
	//{
		//atmosphere *= curve(saturate(originalRayDir.y + 1.0));
	//}


	return atmosphere;
}

vec3 AtmosphericScattering(vec3 rayDir, vec3 lightVector, const float mieAmount, float depth)
{
	const float DEG_TO_RAD = PI / 180.0;

	//Scatter constants
	const float K_R = 0.166;
	const float K_M = 0.0025;
	const float E = Energy;
	const vec3 C_R = C_R1;	//Rayleigh scattering coefficients
	const float G_M = -0.85;

	const float R = 1.0;
	const float SCALE_H = 4.0 / (R - R_INNER);
	const float SCALE_L = 1.0 / (R - R_INNER);

	const int NUM_OUT_SCATTER = 10;
	const float FNUM_OUT_SCATTER = 10.0;

	const int NUM_IN_SCATTER = 10;
	const float FNUM_IN_SCATTER = 10.0;

	vec3 eye = vec3(0.0, mix(R_INNER, 1.0, 0.05), 0.0);

	vec3 originalRayDir = rayDir;

	//if (rayDir.y < 0.0)
	{
		//rayDir.y = abs(rayDir.y);
		//rayDir.y *= rayDir.y;
		//rayDir.y = 0.0;
	}

	vec3 up = vec3(0.0, 1.0, 0.0);

	vec2 e = RaySphereIntersection(eye, rayDir, R);
	vec2 eup = RaySphereIntersection(eye, up, R);
	e.y = depth;
	eup.y = depth;


	vec3 atmosphere = in_scatter(eye, rayDir, e, lightVector, mieAmount, 1.0);

	vec3 secondary = in_scatter2(eye, up, eup, lightVector);

	vec3 ambient = C_R2;

	vec3 ground = vec3(0.1, 0.1, 0.1) * 0.05;

	float boosty = saturate(lightVector.y) * 0.90 + 0.10;
	boosty = 1.0 / sin(boosty);

	//atmosphere += dot(secondary, vec3(0.06)) * ambient * boosty;
	//atmosphere += dot(secondary, vec3(0.86)) * ambient;
	//atmosphere += ambient * 0.01;
	//atmosphere *= pow(C_R2, vec3(0.5));
	
	//atmosphere *= vec3(0.9, 0.9, 1.0);
	atmosphere = pow(atmosphere, vec3(ColorCurved));

	//if (originalRayDir.y < 0.0)
	//{
		//atmosphere *= curve(saturate(originalRayDir.y + 1.0));
	//}


	return atmosphere;
}

vec3 AtmosphericScatteringSingle(vec3 rayDir, vec3 lightVector, const float mieAmount)
{
	const float DEG_TO_RAD = PI / 180.0;

	//Scatter constants
	const float K_R = 0.166;
	const float K_M = 0.0025;
	const float E = Energy;
	const vec3 C_R = C_R1;	//Rayleigh scattering coefficients
	const float G_M = -0.85;

	const float R = 1.0;
	const float SCALE_H = 4.0 / (R - R_INNER);
	const float SCALE_L = 1.0 / (R - R_INNER);

	const int NUM_OUT_SCATTER = 10;
	const float FNUM_OUT_SCATTER = 10.0;

	const int NUM_IN_SCATTER = 10;
	const float FNUM_IN_SCATTER = 10.0;

	vec3 eye = vec3(0.0, mix(R_INNER, 1.0, 0.05), 0.0);

	vec3 originalRayDir = rayDir;

	if (rayDir.y < 0.0)
	{
		//rayDir.y = abs(rayDir.y);
		//rayDir.y *= rayDir.y;
		rayDir.y = 0.0;
	}

	vec3 up = vec3(0.0, 1.0, 0.0);

	vec2 e = RaySphereIntersection(eye, rayDir, R);
	vec2 eup = RaySphereIntersection(eye, up, R);


	vec3 atmosphere = in_scatter(eye, rayDir, e, lightVector, mieAmount, 1.0);


	atmosphere = pow(atmosphere, vec3(ColorCurved));

	//if (originalRayDir.y < 0.0)
	//{
		//atmosphere *= curve(saturate(originalRayDir.y + 1.0));
	//}


	return atmosphere;
}