
const mat3 M = mat3(
     0.5, -1.0,  0.5,
    -1.0,  1.0,  0.5,
     0.5,  0.0,  0.0
);

struct SegmentedSplineParams_c5 {
    float coeffsLow[6];     // Coeffs for B-spline between minPoint and midPoint (units of log luminance)
    float coeffsHigh[6];    // Coeffs for B-spline between midPoint and maxPoint (units of log luminance)
    vec2 minPoint;          // {luminance, luminance} Linear extension below this
    vec2 midPoint;          // {luminance, luminance} Linear
    vec2 maxPoint;          // {luminance, luminance} Linear extension above this
    float slopeLow;         // Log-log slope of low linear extension
    float slopeHigh;        // Log-log slope of high linear extension
};

struct SegmentedSplineParams_c9 {
    float coeffsLow[10];    // Coeffs for B-spline between minPoint and midPoint (units of log luminance)
    float coeffsHigh[10];   // Coeffs for B-spline between midPoint and maxPoint (units of log luminance)
    vec2 minPoint;          // {luminance, luminance} Linear extension below this
    vec2 midPoint;          // {luminance, luminance} Linear
    vec2 maxPoint;          // {luminance, luminance} Linear extension above this
    float slopeLow;         // Log-log slope of low linear extension
    float slopeHigh;        // Log-log slope of high linear extension
};

const SegmentedSplineParams_c5 RRT_PARAMS = SegmentedSplineParams_c5(
    float[6] ( -4.0000000000, -4.0000000000, -3.1573765773, -0.4852499958, 1.8477324706, 1.8477324706 ),    // coeffsLow
    float[6] ( -0.7185482425, 2.0810307172, 3.6681241237, 4.0000000000, 4.0000000000, 4.0000000000 ),       // coeffsHigh
    vec2(0.18 * exp2(-15.0), 0.0001),   // minPoint
    vec2(0.18, 4.8),                    // midPoint
    vec2(0.18 * exp2( 18.0), 10000.0),  // maxPoint
    0.0,    // slopeLow
    0.0     // slopeHigh
);

float segmented_spline_c5_fwd(float x, SegmentedSplineParams_c5 params) { // params should default to RRT_PARAMS
    const int N_KNOTS_LOW  = 4;
    const int N_KNOTS_HIGH = 4;

    float logMinPoint = log10(params.minPoint.x);
    float logMidPoint = log10(params.midPoint.x);
    float logMaxPoint = log10(params.maxPoint.x);

    float logx = log10(x <= 0 ? exp2(-14.0) : x);
    float logy;
 
    if(logx <= logMinPoint) {
        logy = logx * params.slopeLow + (log10(params.minPoint.y) - params.slopeLow * logMinPoint);
    } else if((logx > logMinPoint) && (logx < logMidPoint)) {
        float knot_coord = (N_KNOTS_LOW - 1) * (logx - logMinPoint) / (logMidPoint - logMinPoint);
        int j = int(knot_coord);
        float t = knot_coord - j;

        vec3 cf = vec3(params.coeffsLow[j], params.coeffsLow[j + 1], params.coeffsLow[j + 2]);

        vec3 monomials = vec3(t * t, t, 1.0);
        logy = dot(monomials, M * cf);
    } else if((logx >= logMidPoint) && (logx < logMaxPoint)) {
        float knot_coord = (N_KNOTS_HIGH - 1) * (logx - logMidPoint) / (logMaxPoint - logMidPoint);
        int j = int(knot_coord);
        float t = knot_coord - j;

        vec3 cf = vec3(params.coeffsHigh[j], params.coeffsHigh[j + 1], params.coeffsHigh[j + 2]);

        vec3 monomials = vec3(t * t, t, 1.0);
        logy = dot(monomials, M * cf);
    } else {
        logy = logx * params.slopeHigh + (log10(params.maxPoint.y) - params.slopeHigh * logMaxPoint);
    }

    return pow(10.0, logy);
}

const SegmentedSplineParams_c9 ODT_48nits = SegmentedSplineParams_c9(
    float[10] ( -1.6989700043, -1.6989700043, -1.4779000000, -1.2291000000, -0.8648000000, -0.4480000000, 0.0051800000, 0.4511080334, 0.9113744414, 0.9113744414 ),
    float[10] ( 0.5154386965, 0.8470437783, 1.1358000000, 1.3802000000, 1.5197000000, 1.5985000000, 1.6467000000, 1.6746091357, 1.6878733390, 1.6878733390 ),
    vec2(0.18 * exp2(-6.5), 0.02),  // minPoint
    vec2(0.18, 4.8),                // midPoint
    vec2(0.18 * exp2( 6.5), 48.0),  // maxPoint
    0.0,  // slopeLow
    0.04  // slopeHigh
);

float segmented_spline_c9_fwd(float x, SegmentedSplineParams_c9 params) { // params should default to ODT_48nits
    const int N_KNOTS_LOW  = 8;
    const int N_KNOTS_HIGH = 8;

    float logMinPoint = log10(params.minPoint.x);
    float logMidPoint = log10(params.midPoint.x);
    float logMaxPoint = log10(params.maxPoint.x);

    float logx = log10(x <= 0 ? exp2(-14.0) : x);
    float logy;

    if ( logx <= logMinPoint ) {
        logy = logx * params.slopeLow + (log10(params.minPoint.y) - params.slopeLow * logMinPoint);
    } else if (( logx > logMinPoint ) && ( logx < logMidPoint )) {
        float knot_coord = (N_KNOTS_LOW - 1) * (logx - logMinPoint) / (logMidPoint - logMinPoint);
        int j = int(knot_coord);
        float t = knot_coord - j;

        vec3 cf = vec3(params.coeffsLow[j], params.coeffsLow[j + 1], params.coeffsLow[j + 2]);
        
        vec3 monomials = vec3(t * t, t, 1.0);
        logy = dot(monomials, M * cf);
    } else if (( logx >= logMidPoint ) && ( logx < logMaxPoint )) {
        float knot_coord = (N_KNOTS_HIGH - 1) * (logx - logMidPoint) / (logMaxPoint - logMidPoint);
        int j = int(knot_coord);
        float t = knot_coord - j;

        vec3 cf = vec3(params.coeffsHigh[j], params.coeffsHigh[j + 1], params.coeffsHigh[j + 2]);
        
        vec3 monomials = vec3(t * t, t, 1.0);
        logy = dot(monomials, M * cf);
    } else { //if ( logIn >= logMaxPoint ) {
        logy = logx * params.slopeHigh + (log10(params.maxPoint.y) - params.slopeHigh * logMaxPoint);
    }

    return pow(10.0, logy);
}

const mat3 AP0_2_XYZ = mat3(
    0.9525523959, 0.0000000000,  0.0000936786,
    0.3439664498, 0.7281660966, -0.0721325464,
    0.0000000000, 0.0000000000,  1.0088251844
);

const mat3 XYZ_2_AP0 = mat3(
     1.0498110175, 0.0000000000, -0.0000974845,
    -0.4959030231, 1.3733130458,  0.0982400361,
     0.0000000000, 0.0000000000,  0.9912520182
);

const mat3 AP1_2_XYZ = mat3(
     0.6624541811, 0.1340042065, 0.1561876870,
     0.2722287168, 0.6740817658, 0.0536895174,
    -0.0055746495, 0.0040607335, 1.0103391003
);

const mat3 XYZ_2_AP1 = mat3(
     1.6410233797, -0.3248032942, -0.2364246952,
    -0.6636628587,  1.6153315917,  0.0167563477,
     0.0117218943, -0.0082844420,  0.9883948585
);

const mat3 AP0_2_AP1 = mat3(
     1.4514393161, -0.2365107469, -0.2149285693,
    -0.0765537734,  1.1762296998, -0.0996759264,
     0.0083161484, -0.0060324498,  0.9977163014
);

const mat3 AP1_2_AP0 = mat3(
     0.6954522414, 0.1406786965, 0.1638690622,
     0.0447945634, 0.8596711185, 0.0955343182,
    -0.0055258826, 0.0040252103, 1.0015006723
);

const mat3 sRGB_2_XYZ = mat3(
    0.41239089, 0.35758430, 0.18048084,
    0.21263906, 0.71516860, 0.07219233,
    0.01933082, 0.11919472, 0.95053232
);

const mat3 sRGB_2_ACES = mat3(
    0.43963298, 0.38298870, 0.17737832,
    0.08977644, 0.81343943, 0.09678413,
    0.01754117, 0.11154655, 0.87091228
);

const mat3 XYZ_2_sRGB = mat3(
     3.24096942, -1.53738296, -0.49861076,
    -0.96924388,  1.87596786,  0.04155510,
     0.05563002, -0.20397684,  1.05697131
);

const mat3 D65_2_D60_CAT = mat3(
     1.01303000, 0.00610531, -0.01497100,
     0.00769823, 0.99816500, -0.00503203,
    -0.00284131, 0.00468516,  0.92450700
);

const mat3 D60_2_D65_CAT = mat3(
     0.98722400, -0.00611327, 0.01595330,
    -0.00759836,  1.00186000, 0.00533002,
     0.00307257, -0.00509595, 1.08168000
);

const vec3 AP1_RGB2Y = vec3(AP1_2_XYZ[0][1], AP1_2_XYZ[1][1], AP1_2_XYZ[2][1]);

float rgb_2_saturation(vec3 rgb) {
    const float TINY = 1e-10;
    return (max(TINY, max(max(rgb.r, rgb.g), rgb.b)) - max(min(min(rgb.r, rgb.g), rgb.b), TINY)) / max(max(max(rgb.r, rgb.g), rgb.b), 1e-2);
}

float rgb_2_yc(vec3 rgb, float ycRadiusWeight) { // ycRadiusWeight should default to 1.75
    float chroma = sqrt(rgb.b * (rgb.b - rgb.g) + rgb.g * (rgb.g - rgb.r) + rgb.r * (rgb.r - rgb.b));
    return (rgb.b + rgb.g + rgb.r + ycRadiusWeight * chroma) / 3.0;
}

float rgb_2_hue(vec3 rgb) {
    float hue;
    if(rgb.r == rgb.g && rgb.g == rgb.b) {
        return 0.0; // RGB triplets where RGB are equal have an undefined hue, so return 0
    } else {
        hue = (180.0 * rPI) * atan(sqrt(3) * (rgb.g - rgb.b), 2.0 * rgb.r - rgb.g - rgb.b);
    }
    if(hue < 0.0) hue += 360.0;
    return hue;
}

vec3 XYZ_2_xyY(vec3 XYZ) {
    vec3 xyY;
    float divisor = (XYZ.x + XYZ.y + XYZ.z);
    if (divisor == 0.) divisor = 1e-10;
    xyY.x = XYZ.x / divisor;
    xyY.y = XYZ.y / divisor;
    xyY.z = XYZ.y;
    return xyY;
}

vec3 xyY_2_XYZ(vec3 xyY) {
    vec3 XYZ;
    XYZ.x = xyY.x * xyY.z / max(xyY.y, 1e-10);
    XYZ.y = xyY.z;
    XYZ.z = (1.0 - xyY.x - xyY.y) * xyY.z / max(xyY.y, 1e-10);
    return XYZ;
}

const float sqrt3over4 = 0.433012701892219;  // sqrt(3.)/4.
const mat3 RGB_2_YAB_MAT = mat3(
    1.0 / 3.0,  1.0 / 2.0,  0.0,
    1.0 / 3.0, -1.0 / 4.0,  sqrt3over4,
    1.0 / 3.0, -1.0 / 4.0, -sqrt3over4
);

vec3 rgb_2_yab(vec3 rgb) {
    vec3 yab = RGB_2_YAB_MAT * rgb;

    return yab;
}

mat3 aces_inverse_mat3x3(mat3 m) {
    float a00 = m[0][0], a01 = m[0][1], a02 = m[0][2];
    float a10 = m[1][0], a11 = m[1][1], a12 = m[1][2];
    float a20 = m[2][0], a21 = m[2][1], a22 = m[2][2];

    float b01 = a22 * a11 - a12 * a21;
    float b11 = -a22 * a10 + a12 * a20;
    float b21 = a21 * a10 - a11 * a20;

    float det = a00 * b01 + a01 * b11 + a02 * b21;

    return mat3(b01, (-a22 * a01 + a02 * a21), (a12 * a01 - a02 * a11),
                b11, (a22 * a00 - a02 * a20), (-a12 * a00 + a02 * a10),
                b21, (-a21 * a00 + a01 * a20), (a11 * a00 - a01 * a10)) / det;
}

vec3 yab_2_rgb(vec3 yab) {
    vec3 rgb = aces_inverse_mat3x3(RGB_2_YAB_MAT) * yab;

    return rgb;
}

vec3 yab_2_ych(vec3 yab) {
    vec3 ych = yab;

    ych[1] = sqrt( pow( yab[1], 2.) + pow( yab[2], 2.) );

    ych[2] = atan( yab[2], yab[1] ) * (180.0 * rPI);
    if (ych[2] < 0.0) ych[2] = ych[2] + 360.;

    return ych;
}

vec3 ych_2_yab(vec3 ych) {
    vec3 yab;
    yab[0] = ych[0];

    float h = ych[2] * (PI / 180.0);
    yab[1] = ych[1]*cos(h);
    yab[2] = ych[1]*sin(h);

    return yab;
}

vec3 rgb_2_ych(vec3 rgb) {
    return yab_2_ych( rgb_2_yab( rgb));
}

vec3 ych_2_rgb(vec3 ych) {
    return yab_2_rgb( ych_2_yab( ych));
}

mat3 calc_sat_adjust_matrix(float sat, vec3 rgb2Y) {
    mat3 M;
    M[0][0] = (1.0 - sat) * rgb2Y[0] + sat;
    M[1][0] = (1.0 - sat) * rgb2Y[0];
    M[2][0] = (1.0 - sat) * rgb2Y[0];
    
    M[0][1] = (1.0 - sat) * rgb2Y[1];
    M[1][1] = (1.0 - sat) * rgb2Y[1] + sat;
    M[2][1] = (1.0 - sat) * rgb2Y[1];
    
    M[0][2] = (1.0 - sat) * rgb2Y[2];
    M[1][2] = (1.0 - sat) * rgb2Y[2];
    M[2][2] = (1.0 - sat) * rgb2Y[2] + sat;

    return M;
}

float center_hue(float hue, float centerH) {
    float hueCentered = hue - centerH;
    if(hueCentered < -180.0)      hueCentered += 360.0;
    else if (hueCentered > 180.0) hueCentered -= 360.0;
    return hueCentered;
}

float uncenter_hue(float hueCentered, float centerH) {
    float hue = hueCentered + centerH;
    if (hue < 0.0)        hue += 360.0;
    else if (hue > 360.0) hue -= 360.0;
    return hue;
}

float cubic_basis_shaper(float x, float w) {
    const mat4 M = mat4(
        -1.0 / 6.0,  3.0 / 6.0, -3.0 / 6.0,  1.0 / 6.0,
         3.0 / 6.0, -6.0 / 6.0,  3.0 / 6.0,  0.0 / 6.0,
        -3.0 / 6.0,  0.0 / 6.0,  3.0 / 6.0,  0.0 / 6.0,
         1.0 / 6.0,  4.0 / 6.0,  1.0 / 6.0,  0.0 / 6.0
    );

    float knots[5] = float[5] (
        w * -0.5,
        w * -0.25,
        0.0,
        w *  0.25,
        w *  0.5
    );

    float y = 0;
    if((x > knots[0]) && (x < knots[4])) {
        float knot_coord = (x - knots[0]) * 4.0 / w;
        int j = int(knot_coord);
        float t = knot_coord - j;

        vec4 monomials = vec4(t * t * t, t * t, t, 1.0);

        switch(j) {
            case 3:  y = monomials[0] * M[0][0] + monomials[1] * M[1][0] + monomials[2] * M[2][0] + monomials[3] * M[3][0]; break;
            case 2:  y = monomials[0] * M[0][1] + monomials[1] * M[1][1] + monomials[2] * M[2][1] + monomials[3] * M[3][1]; break;
            case 1:  y = monomials[0] * M[0][2] + monomials[1] * M[1][2] + monomials[2] * M[2][2] + monomials[3] * M[3][2]; break;
            case 0:  y = monomials[0] * M[0][3] + monomials[1] * M[1][3] + monomials[2] * M[2][3] + monomials[3] * M[3][3]; break;
            default: y = 0.0; break;
        }
    }

    return y * 3.0 / 2.0;
}

vec3 sRGB_to_ACES(vec3 color_sRGB) {
    return color_sRGB * sRGB_2_ACES;
}

const float X_BRK = 0.0078125;
const float Y_BRK = 0.155251141552511;
const float A = 10.5402377416545;
const float B = 0.0729055341958355;

vec3 sRGB_to_ACEScct(vec3 color_sRGB) {
    vec3 color_ACEScct = vec3(0.0);
    for(int i = 0; i < 3; ++i) {
        color_ACEScct[i] = color_sRGB[i] <= X_BRK ? A * color_sRGB[i] + B : (log2(color_sRGB[i]) + 9.72) / 17.52;
    }
    return color_ACEScct;
}

vec3 ACEScct_to_sRGB(vec3 color_ACEScct) {
    vec3 color_sRGB = vec3(0.0);
    for(int i = 0; i < 3; ++i) {
        color_sRGB[i] = color_ACEScct[i] > Y_BRK ? exp2(color_ACEScct[i] * 17.52 - 9.72) : (color_ACEScct[i] - B) / A;
    }
    return color_sRGB;
}

vec3 ACES_to_ACEScct(vec3 color_ACES) {
    vec3 color_RGB = color_ACES * AP0_2_AP1;
    return sRGB_to_ACEScct(color_RGB);
}

vec3 ACEScct_to_ACES(vec3 color_ACEScct) {
    vec3 color_sRGB = ACEScct_to_sRGB(color_ACEScct);
    return color_sRGB * AP1_2_AP0;
}

vec3 LMT_ASC_CDL(vec3 color_ACES, vec3 slope, vec3 offset, vec3 power, float saturation) {
    /*
        Slope  -> Gain
        Offset -> Lift
        Power  -> Gamma
    */

    // ACES -> ACEScct
    vec3 color_ACEScct = ACES_to_ACEScct(color_ACES);

    // Slope, Offset, Power
    color_ACEScct = pow((color_ACEScct * slope) + offset, power);

    // Saturation
    float luma = dot(color_ACEScct, vec3(0.2126, 0.7152, 0.0722));
    float satClamp = max(saturation, 0.0);
    color_ACEScct = (color_ACEScct - luma) * satClamp + luma;

    // ACEScct -> ACES
    return ACEScct_to_ACES(color_ACEScct);
}

vec3 LMT_Gamma_Adjust_Linear(vec3 color_ACES, float gamma, float pivot) {
    // Treat as contrast adjustment
    float scalar = pivot / pow(pivot, gamma);
    return pow(color_ACES, vec3(gamma)) * scalar;
}

float interpolate1D(vec2 table[2], float p) {
    if( p <= table[0].x ) return table[0].y;
    if( p >= table[2-1].x ) return table[2-1].y;
    
    for(int i = 0; i < 2 - 1; ++i) {
        if( table[i].x <= p && p < table[i+1].x ) {
            float s = (p - table[i].x) / (table[i+1].x - table[i].x);
            return table[i].y * ( 1 - s ) + table[i+1].y * s;
        }
    }

    return 0.0;
}

vec3 rotate_H_in_H(vec3 color_ACES, float centerH, float widthH, float degreesShift) {
    vec3 color_YCH = rgb_2_ych(color_ACES);
    vec3 color_YCHnew = color_YCH;

    float centeredHue = center_hue(color_YCH.z, centerH);
    float f_H = cubic_basis_shaper(centeredHue, widthH);

    float oldHue = centeredHue;
    float newHue = centeredHue + degreesShift;
    
    float blendedHue = mix(oldHue, newHue, f_H);

    if(f_H > 0.0) color_YCHnew.z = uncenter_hue(blendedHue, centerH);
    
    return ych_2_rgb(color_YCHnew);
}

vec3 scale_C_at_H(vec3 rgb, float centerH, float widthH, float percentC) {
    vec3 new_rgb = rgb;
    vec3 ych = rgb_2_ych( rgb);

    if (ych.y > 0.0f) {  // Only do the chroma adjustment if pixel is non-neutral
        float centeredHue = center_hue(ych.z, centerH);
        float f_H = cubic_basis_shaper(centeredHue, widthH);

        if (f_H > 0.0f) {
            // Scale chroma in affected hue region
            vec3 new_ych = ych;
            new_ych.y = ych.y * (f_H * (percentC - 1.0) + 1.0);
            new_rgb = ych_2_rgb( new_ych);
        }
    }

    return new_rgb;
}

vec3 scale_Y_at_H(vec3 rgb, float centerH, float widthH, float percentY) {
    vec3 new_rgb = rgb;
    vec3 ych = rgb_2_ych( rgb);

    if (ych.y > 0.0f) {  // Only do the chroma adjustment if pixel is non-neutral
        float centeredHue = center_hue(ych.z, centerH);
        float f_H = cubic_basis_shaper(centeredHue, widthH);

        if (f_H > 0.0f) {
            // Scale chroma in affected hue region
            vec3 new_ych = ych;
            new_ych.x = ych.x * (f_H * (percentY - 1.0) + 1.0);
            new_rgb = ych_2_rgb( new_ych);
        } 
    }

    return new_rgb;
}

vec3 scale_C(vec3 rgb, float percentC) {
    vec3 ych = rgb_2_ych( rgb);
    ych.y = ych.y * percentC;
    
    return ych_2_rgb(ych);
}

vec3 scale_Y(vec3 rgb, float percentY) {
    vec3 ych = rgb_2_ych(rgb);
    ych.x = ych.x * percentY;
    
    return ych_2_rgb(ych);
}

mat3 ChromaticAdaptation(vec2 src_xy, vec2 dst_xy) {
    const mat3 ConeResponse = mat3(
         0.8951,  0.2664, -0.1614,
        -0.7502,  1.7135,  0.0367,
         0.0389, -0.0685,  1.0296
    );
    const mat3 InvConeResponse = mat3(
         0.9869929, -0.1470543, 0.1599627,
         0.4323053,  0.5183603, 0.0492912,
        -0.0085287,  0.0400428, 0.9684867
    );

    vec3 src_XYZ = xyY_2_XYZ(vec3(src_xy, 1.0));
    vec3 dst_XYZ = xyY_2_XYZ(vec3(dst_xy, 1.0));

    vec3 src_coneResp = src_XYZ * ConeResponse;
    vec3 dst_coneResp = dst_XYZ * ConeResponse;

    mat3 VonKriesMat = mat3(
        dst_coneResp[0] / src_coneResp[0], 0.0, 0.0,
        0.0, dst_coneResp[1] / src_coneResp[1], 0.0,
        0.0, 0.0, dst_coneResp[2] / src_coneResp[2]
    );

    return (ConeResponse * VonKriesMat) * InvConeResponse;
}

vec2 PlanckianLocusChromaticity(float Temp) {
    float u = (0.860117757 + 1.54118254e-4 * Temp + 1.28641212e-7 * Temp * Temp) / (1.0f + 8.42420235e-4 * Temp + 7.08145163e-7 * Temp * Temp);
    float v = (0.317398726 + 4.22806245e-5 * Temp + 4.20481691e-8 * Temp * Temp) / (1.0f - 2.89741816e-5 * Temp + 1.61456053e-7 * Temp * Temp);

    float x = 3.0 * u / (2.0 * u - 8.0 * v + 4.0);
    float y = 2.0 * v / (2.0 * u - 8.0 * v + 4.0);

    return vec2(x, y);
}

vec2 D_IlluminantChromaticity(float Temp) {
    // Accurate for 4000K < Temp < 25000K
    // in: correlated color temperature
    // out: CIE 1931 chromaticity
    // Correct for revision of Plank's law
    // This makes 6500 == D65
    Temp *= 1.000556328;

    float x = Temp <= 7000 ?
              0.244063 + ( 0.09911e3 + ( 2.9678e6 - 4.6070e9 / Temp ) / Temp ) / Temp :
              0.237040 + ( 0.24748e3 + ( 1.9018e6 - 2.0064e9 / Temp ) / Temp ) / Temp;

    float y = -3.0 * x * x + 2.87 * x - 0.275;

    return vec2(x, y);
}

vec2 PlanckianIsothermal( float Temp, float Tint ) {
    float u = (0.860117757 + 1.54118254e-4 * Temp + 1.28641212e-7 * Temp * Temp) / (1.0 + 8.42420235e-4 * Temp + 7.08145163e-7 * Temp * Temp);
    float v = (0.317398726 + 4.22806245e-5 * Temp + 4.20481691e-8 * Temp * Temp) / (1.0 - 2.89741816e-5 * Temp + 1.61456053e-7 * Temp * Temp);

    float ud = (-1.13758118e9 - 1.91615621e6 * Temp - 1.53177 * Temp * Temp) / pow(1.41213984e6 + 1189.62 * Temp + Temp * Temp, 2.0);
    float vd = ( 1.97471536e9 - 705674.0 * Temp - 308.607 * Temp * Temp) / pow(6.19363586e6 - 179.456 * Temp + Temp * Temp, 2.0); //don't pow2 this

    vec2 uvd = normalize(vec2(u, v));

    // Correlated color temperature is meaningful within +/- 0.05
    u += -uvd.y * Tint * 0.05;
    v +=  uvd.x * Tint * 0.05;

    float x = 3.0 * u / (2.0 * u - 8.0 * v + 4.0);
    float y = 2.0 * v / (2.0 * u - 8.0 * v + 4.0);

    return vec2(x, y);
}

vec3 WhiteBalance(vec3 LinearColor, const float WhiteTemp, const float WhiteTint) {
    LinearColor = LinearColor * AP0_2_AP1;

    vec2 SrcWhiteDaylight = D_IlluminantChromaticity(WhiteTemp);
    vec2 SrcWhitePlankian = PlanckianLocusChromaticity(WhiteTemp);

    vec2 SrcWhite = WhiteTemp < 4000 ? SrcWhitePlankian : SrcWhiteDaylight;
    const vec2 D65White = vec2(0.31270,  0.32900);

    // Offset along isotherm
    vec2 Isothermal = PlanckianIsothermal(WhiteTemp, WhiteTint * 0.25) - SrcWhitePlankian;
    SrcWhite += Isothermal;

    mat3x3 WhiteBalanceMat = ChromaticAdaptation(SrcWhite, D65White);
    WhiteBalanceMat = (sRGB_2_XYZ * WhiteBalanceMat) * XYZ_2_sRGB;

    return LinearColor * WhiteBalanceMat * AP1_2_AP0;
}

#define RRT_GLOW_GAIN 0.05
#define RRT_GLOW_MID  0.08

#define RRT_RED_PIVOT 0.03
#define RRT_RED_HUE   0.0
#define RRT_RED_WIDTH 135.0
#define RRT_RED_SCALE 0.83

#define RRT_SAT_FACTOR 1.0

float glow_fwd(float ycIn, float glowGainIn, float glowMid) {
    float glowGainOut;

    if(ycIn <= 2.0 / 3.0 * glowMid) {
        glowGainOut = glowGainIn;
    } else if(ycIn >= 2.0 * glowMid) {
        glowGainOut = 0.0;
    } else {
        glowGainOut = glowGainIn * (glowMid / ycIn - 1.0 * 0.5);
    }
    
    return glowGainOut;
}

float sigmoid_shaper(float x) {
    float t = max(1.0 - abs(x * 0.5), 0.0);
    float y = 1.0 + sign(x) * (1.0 - t * t);
    return y * 0.5;
}

vec3 RRT(vec3 color_ACES) {
    // Glow
    float saturation = rgb_2_saturation(color_ACES);
    float ycIn = rgb_2_yc(color_ACES, 1.75);
    float s = sigmoid_shaper((saturation - 0.4) * 5.0);
    float addedGlow = 1.0 + glow_fwd(ycIn, RRT_GLOW_GAIN * s, RRT_GLOW_MID);

    color_ACES *= addedGlow;

    // Red Mod
    float hue = rgb_2_hue(color_ACES);
    float centeredHue = center_hue(hue, RRT_RED_HUE);
    float hueWeight = cubic_basis_shaper(centeredHue, RRT_RED_WIDTH);

    color_ACES.r += hueWeight * saturation * (RRT_RED_PIVOT - color_ACES.r) * (1.0 - RRT_RED_SCALE);

    // ACES -> RGB
    color_ACES = max(color_ACES, 0.0);
    vec3 color_RGBpre = color_ACES * AP0_2_AP1;
    color_RGBpre = max(color_RGBpre, 0.0);

    // Global Desat
    color_RGBpre = color_RGBpre * calc_sat_adjust_matrix(RRT_SAT_FACTOR, AP1_RGB2Y);

    // Apply tonescale independently in RGB
    vec3 color_RGBpost;
    color_RGBpost.r = segmented_spline_c5_fwd(color_RGBpre.r, RRT_PARAMS);
    color_RGBpost.g = segmented_spline_c5_fwd(color_RGBpre.g, RRT_PARAMS);
    color_RGBpost.b = segmented_spline_c5_fwd(color_RGBpre.b, RRT_PARAMS);

    // RGB -> OCES
    vec3 color_OCES = color_RGBpost * AP1_2_AP0;

    return color_OCES;
}

#define ODT_CINEMA_WHITE 48.0
const float ODT_CINEMA_BLACK = ODT_CINEMA_WHITE / 2400.0;

#define DISP_GAMMA 2.4
#define OFFSET 0.055

#define DIM_SURROUND_GAMMA 0.9811

#define ODT_SAT_FACTOR 0.93

#define L_W 1.0
#define L_B 0.0

const mat3 XYZ_2_REC709_PRI_MAT = mat3(
     3.2409699419, -1.5373831776, -0.4986107603,
    -0.9692436363,  1.8759675015,  0.0415550574,
     0.0556300797, -0.2039769589,  1.0569715142
);

const mat3 XYZ_2_P3D60_PRI_MAT = mat3(
     2.4027414142, -0.8974841639, -0.3880533700,
    -0.8325796487,  1.7692317536,  0.0237127115,
     0.0388233815, -0.0824996856,  1.0363685997
);

vec3 Y_2_linCV(vec3 Y, const float Ymax, const float Ymin) {
    return (Y - Ymin) / (Ymax - Ymin);
}

vec3 ODT_pre(vec3 color_OCES) {
    vec3 color_RGBpre = color_OCES * AP0_2_AP1;

    SegmentedSplineParams_c9 odtParams = ODT_48nits;

    odtParams.minPoint.x = segmented_spline_c5_fwd(odtParams.minPoint.x, RRT_PARAMS);
    odtParams.midPoint.x = segmented_spline_c5_fwd(odtParams.midPoint.x, RRT_PARAMS);
    odtParams.maxPoint.x = segmented_spline_c5_fwd(odtParams.maxPoint.x, RRT_PARAMS);

    // Apply tonescale independently in RGB
    vec3 color_RGBpost;
    color_RGBpost.r = segmented_spline_c9_fwd(color_RGBpre.r, odtParams);
    color_RGBpost.g = segmented_spline_c9_fwd(color_RGBpre.g, odtParams);
    color_RGBpost.b = segmented_spline_c9_fwd(color_RGBpre.b, odtParams);

    // Scale luminance to linear code value
    return Y_2_linCV(color_RGBpost, ODT_CINEMA_WHITE, ODT_CINEMA_BLACK);
}

vec3 darkSurround_to_dimSurround(vec3 linearCV) {
    vec3 XYZ = linearCV * AP1_2_XYZ;

    vec3 xyY = XYZ_2_xyY(XYZ);
    xyY.z = max(xyY.z, 0.0);
    xyY.z = pow(xyY.z, DIM_SURROUND_GAMMA);
    XYZ = xyY_2_XYZ(xyY);

    return XYZ * XYZ_2_AP1;
}

float moncurve_r(float y, const float gamma, const float offs) {
    const float yb = pow(offs * gamma / ((gamma - 1.0) * (1.0 + offs)), gamma);
    const float rs = pow((gamma - 1.0) / offs, gamma - 1.0) * pow((1.0 + offs) / gamma, gamma);
    return y >= yb ? (1.0 + offs) * pow(y, 1.0 / gamma) - offs : y * rs;
}

vec3 ODT_sRGB_100nits_dim(vec3 color_RGB) {
    // Apply gamma adjustment to compensate for dim surround
    color_RGB = darkSurround_to_dimSurround(color_RGB);

    // Apply desat to compensate for luminance difference
    color_RGB = color_RGB * calc_sat_adjust_matrix(ODT_SAT_FACTOR, AP1_RGB2Y);

    // Convert to display primary encoding
    // RGB -> XYZ
    vec3 color_XYZ = color_RGB * AP1_2_XYZ;

    // Apply CAT from ACES white point to assumed observer adapted white point
    color_XYZ = color_XYZ * D60_2_D65_CAT;

    // CIE XYZ -> Display Primaries
    color_RGB = color_XYZ * XYZ_2_REC709_PRI_MAT;

    // Handle out-of-gamut values
    // Clip values <0 or >1 (ie projecting outside the display primaries)
    color_RGB = saturate(color_RGB);

    // Encode linear code values with transfer function
    vec3 outputCV;
    outputCV.r = moncurve_r(color_RGB.r, DISP_GAMMA, OFFSET);
    outputCV.g = moncurve_r(color_RGB.g, DISP_GAMMA, OFFSET);
    outputCV.b = moncurve_r(color_RGB.b, DISP_GAMMA, OFFSET);

    return outputCV;
}

vec3 bt1886_r(vec3 L, const float gamma, const float Lw, const float Lb) {
    float rGamma = 1.0 / gamma;
    float a = pow(pow(Lw, rGamma) - pow(Lb, rGamma), gamma);
    float b = pow(Lb, rGamma) / (pow(Lw, rGamma) - pow(Lb, rGamma));
    return pow(max(L / a, 0.0), vec3(rGamma)) - b;
}

vec3 ODT_Rec709_100nits_dim(vec3 color_RGB) {
    // Apply gamma adjustment to compensate for dim surround
    color_RGB = darkSurround_to_dimSurround(color_RGB);

    // Apply desat to compensate for luminance difference
    color_RGB = color_RGB * calc_sat_adjust_matrix(ODT_SAT_FACTOR, AP1_RGB2Y);

    // Convert to display primary encoding
    // RGB -> XYZ
    vec3 color_XYZ = color_RGB * AP1_2_XYZ;

    // Apply CAT from ACES white point to assumed observer adapted white point
    color_XYZ = color_XYZ * D60_2_D65_CAT;

    // CIE XYZ -> Display Primaries
    color_RGB = color_XYZ * XYZ_2_REC709_PRI_MAT;

    // Handle out-of-gamut values
    // Clip values <0 or >1 (ie projecting outside the display primaries)
    color_RGB = saturate(color_RGB);
    
    // Encode linear code values with transfer function
    return bt1886_r(color_RGB, DISP_GAMMA, L_W, L_B);
}

vec3 LMT(vec3 ACES) { // Out in ACES

    #define SLOPE_R 1.0
    #define SLOPE_G 1.0
    #define SLOPE_B 1.0
    #define SLOPE_M 1.0

    #define OFFSET_R 0.0
    #define OFFSET_G 0.0
    #define OFFSET_B 0.0
    #define OFFSET_M 0.0

    #define POWER_R 1.0
    #define POWER_G 1.0
    #define POWER_B 1.0
    #define POWER_M 1.0

    #define GAMMA_PIVOT 0.18 // Should default to 0.18

    #define WHITE_BALANCE 6500 // [4000 4100 4200 4300 4400 4500 4600 4700 4800 4900 5000 5100 5200 5300 5400 5500 5600 5700 5800 5900 6000 6100 6200 6300 6400 6500 6600 6700 6800 6900 7000 7100 7200 7300 7400 7500 7600 7700 7800 7900 8000 8100 8200 8300 8400 8500 8600 8700 8800 8900 9000 9100 9200 9300 9400 9500 9600 9700 9800 9900 10000 10100 10200 10300 10400 10500 10600 10700 10800 10900 11000 11100 11200 11300 11400 11500 11600 11700 11800 11900 12000]
    #define TINT_BALANCE 0.0 // [-1.0 -0.95 -0.9 -0.85 -0.8 -0.75 -0.7 -0.65 -0.6 -0.55 -0.5 -0.45 -0.4 -0.35 -0.3 -0.25 -0.2 -0.15 -0.1 -0.05 0.0 0.05 0.1 0.15 0.2 0.25 0.3 0.35 0.4 0.45 0.5 0.55 0.6 0.65 0.7 0.75 0.8 0.85 0.9 0.95 1.0]

    const vec3 slope  = vec3(SLOPE_R, SLOPE_G, SLOPE_B) * SLOPE_M;
    const vec3 offset = vec3(OFFSET_R, OFFSET_G, OFFSET_B) + OFFSET_M;
    const vec3 power  = vec3(POWER_R, POWER_G, POWER_B) * POWER_M;
    ACES = LMT_ASC_CDL(ACES, slope, offset, power, SATURATION);
            ACES = LMT_Gamma_Adjust_Linear(ACES, 0.9, GAMMA_PIVOT);
            ACES = WhiteBalance(ACES, 7100.0, -0.32);
            ACES = LMT_ASC_CDL(ACES, vec3(1.0, 1.0, 1.0), vec3(0.0, 0.0, 0.0), vec3(0.92, 0.92, 0.92), 1.0);
            ACES = LMT_ASC_CDL(ACES, vec3(0.98, 0.98, 0.98), vec3(0.0, 0.0, 0.0), vec3(1.0, 1.0097, 1.0) * 0.98, 1.0);	
    ACES = LMT_Gamma_Adjust_Linear(ACES, GAMMA, GAMMA_PIVOT);
    ACES = WhiteBalance(ACES, float(WHITE_BALANCE), TINT_BALANCE);
    
    return ACES;
}